<?php
/**
*
* This file is part of the phpBB Forum Software package.
*
* @copyright (c) phpBB Limited <https://www.phpbb.com>
* @license GNU General Public License, version 2 (GPL-2.0)
*
* For full copyright and license information, please see
* the docs/CREDITS.txt file.
*
*/

namespace phpbb\captcha;

class gd
{
	var $width = 360;
	var $height = 96;

	/**
	* Create the image containing $code with a seed of $seed
	*/
	function execute($code, $seed)
	{
		global $config;

		mt_srand($seed);

		// Create image
		$img = imagecreatetruecolor($this->width, $this->height);

		// Generate colours
		$colour = new colour_manager($img, array(
			'random'	=> true,
			'min_value'	=> 60,
		), 'hsv');

		$scheme = $colour->colour_scheme('background', false);
		$scheme = $colour->mono_range($scheme, 10, false);
		shuffle($scheme);

		$bg_colours = array_splice($scheme, mt_rand(6, 12));

		// Generate code characters
		$characters = $sizes = $bounding_boxes = $noise = array();
		$width_avail = $this->width - 15;
		$code_len = strlen($code);
		$captcha_bitmaps = $this->captcha_bitmaps();

		for ($i = 0; $i < $code_len; ++$i)
		{
			$characters[$i] = new char_cube3d($captcha_bitmaps, $code[$i]);

			list($min, $max) = $characters[$i]->range();
			$sizes[$i] = mt_rand($min, $max);

			$box = $characters[$i]->dimensions($sizes[$i]);
			$width_avail -= ($box[2] - $box[0]);
			$bounding_boxes[$i] = $box;
		}

		// Redistribute leftover x-space
		$offset = array();
		for ($i = 0; $i < $code_len; ++$i)
		{
			$denom = ($code_len - $i);
			$denom = max(1.3, $denom);
			$offset[$i] = phpbb_mt_rand(0, (int) round((1.5 * $width_avail) / $denom));
			$width_avail -= $offset[$i];
		}

		if ($config['captcha_gd_x_grid'])
		{
			$grid = (int) $config['captcha_gd_x_grid'];
			for ($y = 0; $y < $this->height; $y += mt_rand($grid - 2, $grid + 2))
			{
				$current_colour = $scheme[array_rand($scheme)];
				imageline($img, mt_rand(0,4), mt_rand($y - 3, $y), mt_rand($this->width - 5, $this->width), mt_rand($y - 3, $y), $current_colour);
			}
		}

		if ($config['captcha_gd_y_grid'])
		{
			$grid = (int) $config['captcha_gd_y_grid'];
			for ($x = 0; $x < $this->width; $x += mt_rand($grid - 2, $grid + 2))
			{
				$current_colour = $scheme[array_rand($scheme)];
				imagedashedline($img, mt_rand($x -3, $x + 3), mt_rand(0, 4), mt_rand($x -3, $x + 3), mt_rand($this->height - 5, $this->height), $current_colour);
			}
		}

		if ($config['captcha_gd_wave'] && ($config['captcha_gd_y_grid'] || $config['captcha_gd_y_grid']))
		{
			$this->wave($img);
		}

		if ($config['captcha_gd_3d_noise'])
		{
			$xoffset = mt_rand(0,9);
			$noise_bitmaps = $this->captcha_noise_bg_bitmaps();
			for ($i = 0; $i < $code_len; ++$i)
			{
				$noise[$i] = new char_cube3d($noise_bitmaps, mt_rand(1, sizeof($noise_bitmaps['data'])));

				list($min, $max) = $noise[$i]->range();
				//$box = $noise[$i]->dimensions($sizes[$i]);
			}
			$xoffset = 0;
			for ($i = 0; $i < $code_len; ++$i)
			{
				$dimm = $bounding_boxes[$i];
				$xoffset += ($offset[$i] - $dimm[0]);
				$yoffset = mt_rand(-$dimm[1], $this->height - $dimm[3]);

				$noise[$i]->drawchar($sizes[$i], $xoffset, $yoffset, $img, $colour->get_resource('background'), $scheme);
				$xoffset += $dimm[2];
			}
		}

		$xoffset = 5;
		for ($i = 0; $i < $code_len; ++$i)
		{
			$dimm = $bounding_boxes[$i];
			$xoffset += ($offset[$i] - $dimm[0]);
			$yoffset = mt_rand(-$dimm[1], $this->height - $dimm[3]);

			$characters[$i]->drawchar($sizes[$i], $xoffset, $yoffset, $img, $colour->get_resource('background'), $scheme);
			$xoffset += $dimm[2];
		}

		if ($config['captcha_gd_wave'])
		{
			$this->wave($img);
		}

		if ($config['captcha_gd_foreground_noise'])
		{
			$this->noise_line($img, 0, 0, $this->width, $this->height, $colour->get_resource('background'), $scheme, $bg_colours);
		}

		// Send image
		header('Content-Type: image/png');
		header('Cache-control: no-cache, no-store');
		imagepng($img);
		imagedestroy($img);
	}

	/**
	* Sinus
	*/
	function wave($img)
	{
		global $config;

		$period_x = mt_rand(12,18);
		$period_y = mt_rand(7,14);
		$amp_x = mt_rand(5,10);
		$amp_y = mt_rand(2,4);
		$socket = mt_rand(0,100);

		$dampen_x = mt_rand($this->width/5, $this->width/2);
		$dampen_y = mt_rand($this->height/5, $this->height/2);
		$direction_x = (mt_rand (0, 1));
		$direction_y = (mt_rand (0, 1));

		for ($i = 0; $i < $this->width; $i++)
		{
			$dir = ($direction_x) ? $i : ($this->width - $i);
			imagecopy($img, $img, $i-1, sin($socket+ $i/($period_x + $dir/$dampen_x)) * $amp_x, $i, 0, 1, $this->height);
		}
		$socket = mt_rand(0,100);
		for ($i = 0; $i < $this->height; $i++)
		{
			$dir = ($direction_y) ? $i : ($this->height - $i);
			imagecopy($img, $img ,sin($socket + $i/($period_y + ($dir)/$dampen_y)) * $amp_y, $i-1, 0, $i, $this->width, 1);
		}
		return $img;
	}

	/**
	* Noise line
	*/
	function noise_line($img, $min_x, $min_y, $max_x, $max_y, $bg, $font, $non_font)
	{
		imagesetthickness($img, 2);

		$x1 = $min_x;
		$x2 = $max_x;
		$y1 = $min_y;
		$y2 = $min_y;

		do
		{
			$line = array_merge(
				array_fill(0, mt_rand(30, 60), $non_font[array_rand($non_font)]),
				array_fill(0, mt_rand(30, 60), $bg)
			);

			imagesetstyle($img, $line);
			imageline($img, $x1, $y1, $x2, $y2, IMG_COLOR_STYLED);

			$y1 += mt_rand(12, 35);
			$y2 += mt_rand(12, 35);
		}
		while ($y1 < $max_y && $y2 < $max_y);

		$x1 = $min_x;
		$x2 = $min_x;
		$y1 = $min_y;
		$y2 = $max_y;

		do
		{
			$line = array_merge(
				array_fill(0, mt_rand(30, 60), $non_font[array_rand($non_font)]),
				array_fill(0, mt_rand(30, 60), $bg)
			);

			imagesetstyle($img, $line);
			imageline($img, $x1, $y1, $x2, $y2, IMG_COLOR_STYLED);

			$x1 += mt_rand(20, 35);
			$x2 += mt_rand(20, 35);
		}
		while ($x1 < $max_x && $x2 < $max_x);
		imagesetthickness($img, 1);
	}

	function captcha_noise_bg_bitmaps()
	{
		return array(
			'width'		=> 15,
			'height'	=> 5,
			'data'		=> array(

			1 => array(
				array(1,0,0,0,1,0,0,0,0,0,0,0,0,0,0),
				array(1,0,0,0,0,1,0,0,0,0,0,0,0,0,0),
				array(1,0,0,0,0,0,0,0,0,0,0,0,0,0,0),
				array(1,0,0,0,0,1,0,0,0,0,0,0,1,0,0),
				array(1,0,0,0,0,0,1,0,0,0,0,1,0,0,0),
			),
			2 => array(
				array(1,1,mt_rand(0,1),1,0,1,1,1,1,0,0,0,0,0,0),
				array(0,0,0,0,0,0,0,1,0,0,0,0,0,0,0),
				array(0,0,0,0,0,0,0,0,0,0,0,0,0,0,0),
				array(0,0,0,0,0,0,0,0,0,0,0,0,0,0,0),
				array(0,0,0,0,0,0,0,0,0,1,1,0,1,1,1),
			),
			3 => array(
				array(1,0,0,0,0,0,0,0,0,0,0,0,0,0,1),
				array(1,0,0,0,0,0,0,0,0,0,0,0,0,1,0),
				array(0,0,0,0,1,0,0,0,0,0,0,0,0,0,1),
				array(1,0,0,0,0,0,0,0,0,0,0,0,0,1,0),
				array(1,0,0,0,0,0,0,0,0,0,0,0,0,0,1),
			),
			4 => array(
				array(1,0,1,0,1,0,0,1,1,0,0,0,0,0,0),
				array(0,0,0,0,0,0,0,1,0,0,0,0,0,0,0),
				array(0,0,0,0,0,0,0,0,0,0,0,0,0,0,0),
				array(0,0,0,0,0,0,0,0,0,0,0,0,0,0,0),
				array(1,0,1,0,0,0,0,0,0,0,0,0,0,0,0),
			),
			5 => array(
				array(1,1,1,1,0,0,0,1,1,1,0,0,1,0,1),
				array(0,0,0,0,0,0,0,1,0,0,0,0,0,0,0),
				array(0,0,0,0,0,0,0,0,0,0,0,0,0,0,0),
				array(0,0,0,0,0,0,0,0,0,0,0,0,0,0,0),
				array(1,0,1,0,0,0,0,0,0,0,0,0,0,0,0),
			),
			6 => array(
				array(mt_rand(0,1),mt_rand(0,1),mt_rand(0,1),mt_rand(0,1),mt_rand(0,1),0,mt_rand(0,1),mt_rand(0,1),mt_rand(0,1),mt_rand(0,1),mt_rand(0,1),0,mt_rand(0,1),mt_rand(0,1),mt_rand(0,1)),
				array(0,0,0,0,0,0,0,mt_rand(0,1),0,0,0,0,0,0,0),
				array(0,0,0,0,0,0,0,0,0,0,0,0,0,0,0),
				array(0,0,0,0,0,0,0,0,0,0,0,0,0,0,0),
				array(mt_rand(0,1),0,mt_rand(0,1),0,0,0,0,0,0,0,0,0,0,0,0),
			),
			7 => array(
				array(0,0,0,0,0,0,0,0,0,0,1,1,0,1,1),
				array(0,0,0,0,0,0,0,0,0,0,0,0,0,0,0),
				array(0,0,1,1,0,0,0,1,0,0,0,0,0,0,0),
				array(0,1,0,0,0,1,0,0,0,0,0,0,0,0,0),
				array(1,0,0,0,0,0,0,0,0,0,0,0,0,0,0),
			),
		));
	}

	/**
	* Return bitmaps
	*/
	function captcha_bitmaps()
	{
		global $config;

		$chars = array(
			'A'	=>	array(
						array(
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,1,0,1,0,0,0),
							array(0,0,0,1,0,1,0,0,0),
							array(0,0,0,1,0,1,0,0,0),
							array(0,0,1,0,0,0,1,0,0),
							array(0,0,1,0,0,0,1,0,0),
							array(0,0,1,0,0,0,1,0,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,1,1,1,1,1,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
						),
						array(
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,1,0,1,0,0,0),
							array(0,0,1,1,0,1,1,0,0),
							array(0,0,1,0,0,0,1,0,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,1,1,1,1,1,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(1,1,1,0,0,0,1,1,1),
						),
						array(
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,1,1,1,1,1,0,0),
							array(0,1,1,0,0,0,1,1,0),
							array(1,1,0,0,0,0,0,1,1),
							array(1,0,0,0,0,0,0,0,1),
							array(0,0,0,0,0,0,0,1,1),
							array(0,0,0,0,0,1,1,1,1),
							array(0,0,0,1,1,1,0,0,1),
							array(0,1,1,1,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,1,0,0,0,0,1,1,1),
							array(0,1,1,1,1,1,1,0,1),
						),
					),
		'B'	=>		array(
						array(
							array(1,1,1,1,1,1,1,0,0),
							array(1,0,0,0,0,0,0,1,0),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,1,0),
							array(1,1,1,1,1,1,1,0,0),
							array(1,0,0,0,0,0,0,1,0),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,1,0),
							array(1,1,1,1,1,1,1,0,0),
						),
						array(
							array(1,1,1,1,1,1,1,0,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,1,1,1,1,1,0,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,1,0),
							array(1,1,1,1,1,1,1,0,0),
						),
						array(
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,1,1,1,1,1,0,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,1,1,1,1,1,0,0),
						),
					),
		'C'	=>		array(
						array(
							array(0,0,1,1,1,1,1,0,0),
							array(0,1,0,0,0,0,0,1,0),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,1,0),
							array(0,0,1,1,1,1,1,0,0),
						),
						array(
							array(0,0,1,1,1,1,1,0,1),
							array(0,1,0,0,0,0,0,1,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,1,1),
							array(0,0,1,1,1,1,1,0,1),
						),
					),
		'D'	=>		array(
						array(
							array(1,1,1,1,1,1,1,0,0),
							array(1,0,0,0,0,0,0,1,0),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,1,0),
							array(1,1,1,1,1,1,1,0,0),
						),
						array(
							array(1,1,1,1,1,1,1,0,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,1,0),
							array(1,1,1,1,1,1,1,0,0),
						),
						array(
							array(0,0,0,0,0,0,0,0,1),
							array(0,0,0,0,0,0,0,0,1),
							array(0,0,0,0,0,0,0,0,1),
							array(0,0,0,0,0,0,0,0,1),
							array(0,0,0,0,0,0,0,0,1),
							array(0,0,0,0,0,0,0,0,1),
							array(0,0,0,0,0,0,0,0,1),
							array(0,0,1,1,1,1,1,0,1),
							array(0,1,1,0,0,0,1,1,1),
							array(0,1,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,0,1),
							array(0,1,1,0,0,0,1,1,1),
							array(0,0,1,1,1,1,1,0,1),
						),
					),
		'E'	=>		array(
						array(
							array(1,1,1,1,1,1,1,1,1),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,1,1,1,1,1,1,1,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,1,1,1,1,1,1,1,1),
						),
						array(
							array(1,1,1,1,1,1,1,1,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,1,1,1,1,1,1,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,1),
							array(1,1,1,1,1,1,1,1,1),
						),
						array(
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,1,1,1,1,1,0,0),
							array(0,1,1,0,0,0,1,1,0),
							array(1,1,0,0,0,0,0,1,1),
							array(1,1,1,1,1,1,1,1,1),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,1),
							array(1,1,0,0,0,0,0,1,1),
							array(0,1,1,1,1,1,1,1,0),
						),
					),
		'F'	=>		array(
						array(
							array(1,1,1,1,1,1,1,1,1),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,1,1,1,1,1,1,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
						),
						array(
							array(0,1,1,1,1,1,1,1,1),
							array(0,1,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,1,1,1,1,1,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(1,1,1,0,0,0,0,0,0),
						),
						array(
							array(0,0,0,1,1,0,0,0,0),
							array(0,0,1,1,0,0,0,0,0),
							array(0,1,1,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(1,1,1,1,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
						),
					),
		'G'	=>		array(
						array(
							array(0,0,1,1,1,1,1,0,0),
							array(0,1,0,0,0,0,0,1,0),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,1,1,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,1,0),
							array(0,0,1,1,1,1,1,0,0),
						),
						array(
							array(0,0,1,1,1,1,1,0,1),
							array(0,1,0,0,0,0,0,1,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,1,1,1,1,1),
							array(1,0,0,0,1,0,0,0,1),
							array(1,0,0,0,1,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,1,1),
							array(0,0,1,1,1,1,1,0,1),
						),
						array(
							array(0,0,1,1,1,1,1,0,1),
							array(0,1,1,0,0,0,0,1,1),
							array(1,1,0,0,0,0,0,1,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,1,1,0,0,0,0,0,1),
							array(0,0,1,1,1,1,1,1,1),
							array(0,0,0,0,0,0,0,0,1),
							array(0,0,0,0,0,0,0,0,1),
							array(0,0,0,0,0,0,0,0,1),
							array(0,0,0,0,0,0,0,0,1),
							array(0,0,0,0,0,0,0,1,1),
							array(1,1,1,1,1,1,1,1,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
						),
					),
		'H'	=>		array(
						array(
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,1,1,1,1,1,1,1,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
						),
						array(
							array(1,1,1,0,0,0,1,1,1),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,1,1,1,1,1,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(1,1,1,0,0,0,1,1,1),
						),
						array(
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,1,1,1,0,0,0),
							array(1,1,1,1,0,1,1,0,0),
							array(1,0,0,0,0,0,1,0,0),
							array(1,0,0,0,0,0,1,0,0),
							array(1,0,0,0,0,0,1,0,0),
							array(1,0,0,0,0,0,1,0,0),
							array(1,0,0,0,0,0,1,0,0),
							array(1,0,0,0,0,0,1,0,0),
						),
					),
		'I'	=>		array(
						array(
							array(1,1,1,1,1,1,1,1,1),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(1,1,1,1,1,1,1,1,1),
						),
						array(
							array(0,0,0,1,1,1,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,1,1,1,0,0,0),
						),
						array(
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,1,1,1,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,1,1,1,0,0,0),
						),
					),
		'J'	=>		array(
						array(
							array(1,1,1,1,1,1,1,1,1),
							array(0,0,0,0,0,1,0,0,0),
							array(0,0,0,0,0,1,0,0,0),
							array(0,0,0,0,0,1,0,0,0),
							array(0,0,0,0,0,1,0,0,0),
							array(0,0,0,0,0,1,0,0,0),
							array(0,0,0,0,0,1,0,0,0),
							array(0,0,0,0,0,1,0,0,0),
							array(0,0,0,0,0,1,0,0,0),
							array(0,0,0,0,0,1,0,0,0),
							array(0,0,0,0,0,1,0,0,0),
							array(1,0,0,0,0,1,0,0,0),
							array(1,0,0,0,0,1,0,0,0),
							array(0,1,0,0,1,0,0,0,0),
							array(0,0,1,1,0,0,0,0,0),
						),
						array(
							array(1,1,1,1,1,1,1,1,1),
							array(0,0,0,0,0,1,0,0,0),
							array(0,0,0,0,0,1,0,0,0),
							array(0,0,0,0,0,1,0,0,0),
							array(0,0,0,0,0,1,0,0,0),
							array(0,0,0,0,0,1,0,0,0),
							array(0,0,0,0,0,1,0,0,0),
							array(0,0,0,0,0,1,0,0,0),
							array(0,0,0,0,0,1,0,0,0),
							array(0,0,0,0,0,1,0,0,0),
							array(0,0,0,0,0,1,0,0,0),
							array(1,0,0,0,0,1,0,0,0),
							array(1,0,0,0,0,1,0,0,0),
							array(1,1,0,0,1,0,0,0,0),
							array(1,0,1,1,0,0,0,0,0),
						),
						array(
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,1,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,1,0,0,0),
							array(0,0,0,0,0,1,0,0,0),
							array(0,0,0,0,0,1,0,0,0),
							array(0,0,0,0,0,1,0,0,0),
							array(0,0,0,0,0,1,0,0,0),
							array(1,0,0,0,0,1,0,0,0),
							array(1,0,0,0,0,1,0,0,0),
							array(0,1,0,0,1,0,0,0,0),
							array(0,0,1,1,0,0,0,0,0),
						),
					),
		'K'	=>		array(
						array(    // New 'K', supplied by NeoThermic
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,1,0),
							array(1,0,0,0,0,0,1,0,0),
							array(1,0,0,0,0,1,0,0,0),
							array(1,0,0,0,1,0,0,0,0),
							array(1,0,0,1,0,0,0,0,0),
							array(1,0,1,0,0,0,0,0,0),
							array(1,1,0,0,0,0,0,0,0),
							array(1,0,1,0,0,0,0,0,0),
							array(1,0,0,1,0,0,0,0,0),
							array(1,0,0,0,1,0,0,0,0),
							array(1,0,0,0,0,1,0,0,0),
							array(1,0,0,0,0,0,1,0,0),
							array(1,0,0,0,0,0,0,1,0),
							array(1,0,0,0,0,0,0,0,1),
						),
						array(
							array(0,1,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,1,0,0),
							array(0,1,0,0,0,1,0,0,0),
							array(0,1,0,0,1,0,0,0,0),
							array(0,1,0,1,0,0,0,0,0),
							array(0,1,1,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,1,0,0,0,0,0,0),
							array(0,1,0,1,0,0,0,0,0),
							array(0,1,0,0,1,0,0,0,0),
							array(0,1,0,0,0,1,0,0,0),
							array(0,1,0,0,0,0,1,0,0),
							array(0,1,0,0,0,0,0,1,0),
							array(1,1,1,0,0,0,1,1,1),
						),
						array(
							array(0,0,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,1,0,0,0),
							array(0,1,0,0,1,0,0,0,0),
							array(0,1,0,1,0,0,0,0,0),
							array(0,1,1,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,1,0,0,0,0,0,0),
							array(0,1,0,1,0,0,0,0,0),
							array(0,1,0,0,1,0,0,0,0),
							array(0,1,0,0,0,1,0,0,0),
							array(0,1,0,0,0,0,1,0,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
						),
					),
		'L'	=>		array(
						array(
							array(0,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,1,1,1,1,1,1,1,1),
						),
						array(
							array(0,0,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,1),
							array(1,1,1,1,1,1,1,1,1),
						),
						array(
							array(0,0,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,1,0,0,0,0,0,0),
							array(0,0,1,1,1,0,0,0,0),
						),
					),
		'M'	=>		array(
						array(
							array(1,1,0,0,0,0,0,1,1),
							array(1,1,0,0,0,0,0,1,1),
							array(1,0,1,0,0,0,1,0,1),
							array(1,0,1,0,0,0,1,0,1),
							array(1,0,1,0,0,0,1,0,1),
							array(1,0,0,1,0,1,0,0,1),
							array(1,0,0,1,0,1,0,0,1),
							array(1,0,0,1,0,1,0,0,1),
							array(1,0,0,0,1,0,0,0,1),
							array(1,0,0,0,1,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
						),
						array(
							array(0,0,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,1,0,0,0,1,1,0),
							array(0,1,1,0,0,0,1,1,0),
							array(0,1,1,0,0,0,1,1,0),
							array(0,1,0,1,0,1,0,1,0),
							array(0,1,0,1,0,1,0,1,0),
							array(0,1,0,1,0,1,0,1,0),
							array(0,1,0,0,1,0,0,1,0),
							array(0,1,0,0,1,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(1,1,1,0,0,0,1,1,1),
						),
						array(
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,1,1,1,0,1,1,1,0),
							array(1,1,0,1,1,1,0,1,1),
							array(1,0,0,0,1,0,0,0,1),
							array(1,0,0,0,1,0,0,0,1),
							array(1,0,0,0,1,0,0,0,1),
							array(1,0,0,0,1,0,0,0,1),
							array(1,0,0,0,1,0,0,0,1),
							array(1,0,0,0,1,0,0,0,1),
							array(1,0,0,0,1,0,0,0,1),
						),
					),
		'N'	=>		array(
						array(
							array(1,1,0,0,0,0,0,0,1),
							array(1,1,0,0,0,0,0,0,1),
							array(1,0,1,0,0,0,0,0,1),
							array(1,0,1,0,0,0,0,0,1),
							array(1,0,0,1,0,0,0,0,1),
							array(1,0,0,1,0,0,0,0,1),
							array(1,0,0,0,1,0,0,0,1),
							array(1,0,0,0,1,0,0,0,1),
							array(1,0,0,0,1,0,0,0,1),
							array(1,0,0,0,0,1,0,0,1),
							array(1,0,0,0,0,1,0,0,1),
							array(1,0,0,0,0,0,1,0,1),
							array(1,0,0,0,0,0,1,0,1),
							array(1,0,0,0,0,0,0,1,1),
							array(1,0,0,0,0,0,0,1,1),
						),
						array(
							array(0,0,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,1,0,0,0,0,1,0),
							array(0,1,1,0,0,0,0,1,0),
							array(0,1,1,0,0,0,0,1,0),
							array(0,1,0,1,0,0,0,1,0),
							array(0,1,0,1,0,0,0,1,0),
							array(0,1,0,1,0,0,0,1,0),
							array(0,1,0,0,1,0,0,1,0),
							array(0,1,0,0,1,1,0,1,0),
							array(0,1,0,0,0,1,0,1,0),
							array(0,1,0,0,0,1,1,1,0),
							array(0,1,0,0,0,0,1,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(1,1,1,0,0,0,1,1,1),
						),
						array(
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(1,0,1,1,1,1,0,0,0),
							array(1,1,1,0,0,1,1,0,0),
							array(1,0,0,0,0,0,1,0,0),
							array(1,0,0,0,0,0,1,0,0),
							array(1,0,0,0,0,0,1,0,0),
							array(1,0,0,0,0,0,1,0,0),
							array(1,0,0,0,0,0,1,0,0),
							array(1,0,0,0,0,0,1,0,0),
						),
					),
		'O'	=>		array(
						array(
							array(0,0,1,1,1,1,1,0,0),
							array(0,1,0,0,0,0,0,1,0),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,1,0),
							array(0,0,1,1,1,1,1,0,0),
						),
						array(
							array(0,0,1,1,1,1,1,0,0),
							array(0,1,0,0,0,0,0,1,0),
							array(1,1,0,0,0,0,0,1,1),
							array(1,1,0,0,0,0,0,1,1),
							array(1,1,0,0,0,0,0,1,1),
							array(1,1,0,0,0,0,0,1,1),
							array(1,1,0,0,0,0,0,1,1),
							array(1,1,0,0,0,0,0,1,1),
							array(1,1,0,0,0,0,0,1,1),
							array(1,1,0,0,0,0,0,1,1),
							array(1,1,0,0,0,0,0,1,1),
							array(1,1,0,0,0,0,0,1,1),
							array(1,1,0,0,0,0,0,1,1),
							array(0,1,0,0,0,0,0,1,0),
							array(0,0,1,1,1,1,1,0,0),
						),
						array(
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,1,1,1,1,1,0,0,0),
							array(1,1,1,0,0,1,1,0,0),
							array(1,0,0,0,0,0,1,0,0),
							array(1,0,0,0,0,0,1,0,0),
							array(1,0,0,0,0,0,1,0,0),
							array(1,0,0,0,0,0,1,0,0),
							array(1,1,0,0,0,1,1,0,0),
							array(0,1,1,1,1,1,0,0,0),
						),
					),
		'P'	=>		array(
						array(
							array(1,1,1,1,1,1,1,0,0),
							array(1,0,0,0,0,0,0,1,0),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,1,0),
							array(1,1,1,1,1,1,1,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
						),
						array(
							array(1,1,1,1,1,1,1,0,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,1,0),
							array(1,1,1,1,1,1,1,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(1,1,1,0,0,0,0,0,0),
						),
						array(
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,1,1,0,0,0,0,0),
							array(1,1,0,1,1,0,0,0,0),
							array(1,0,0,0,1,0,0,0,0),
							array(1,0,0,0,1,0,0,0,0),
							array(1,0,0,1,1,0,0,0,0),
							array(1,1,1,1,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
						),
					),
		'Q'	=>		array(
						array(
							array(0,0,1,1,1,1,1,0,0),
							array(0,1,0,0,0,0,0,1,0),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,1,0,0,1),
							array(1,0,0,0,0,0,1,0,1),
							array(0,1,0,0,0,0,0,1,0),
							array(0,0,1,1,1,1,1,0,1),
						),
						array(
							array(0,0,1,1,1,1,1,0,0),
							array(0,1,0,0,0,0,0,1,0),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,1,0,0,0,1),
							array(1,1,0,0,1,1,0,1,1),
							array(0,1,1,1,1,1,1,1,0),
							array(0,0,0,0,0,0,1,1,0),
							array(0,0,0,0,0,0,0,1,1),
							array(0,0,0,0,0,0,0,0,1),
						),
						array(
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,1,1,1,1),
							array(0,0,0,0,1,1,0,0,1),
							array(0,0,0,0,1,0,0,0,1),
							array(0,0,0,0,1,0,0,0,1),
							array(0,0,0,0,1,1,0,1,1),
							array(0,0,0,0,0,1,1,0,1),
							array(0,0,0,0,0,0,0,0,1),
							array(0,0,0,0,0,0,0,0,1),
							array(0,0,0,0,0,0,0,0,1),
							array(0,0,0,0,0,0,0,0,1),
							array(0,0,0,0,0,0,0,0,1),
							array(0,0,0,0,0,0,0,0,1),
						),
					),
		'R'	=>		array(
						array(
							array(1,1,1,1,1,1,1,0,0),
							array(1,0,0,0,0,0,0,1,0),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,1,0),
							array(1,1,1,1,1,1,1,0,0),
							array(1,1,1,0,0,0,0,0,0),
							array(1,0,0,1,0,0,0,0,0),
							array(1,0,0,0,1,0,0,0,0),
							array(1,0,0,0,0,1,0,0,0),
							array(1,0,0,0,0,0,1,0,0),
							array(1,0,0,0,0,0,0,1,0),
							array(1,0,0,0,0,0,0,0,1),
						),
						array(
							array(1,1,1,1,1,1,1,0,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,1,0),
							array(1,1,1,1,1,1,1,0,0),
							array(0,1,1,0,0,0,0,0,0),
							array(0,1,1,1,0,0,0,0,0),
							array(0,1,0,1,1,0,0,0,0),
							array(0,1,0,0,1,1,0,0,0),
							array(0,1,0,0,0,1,1,0,0),
							array(0,1,0,0,0,0,1,1,0),
							array(1,1,1,0,0,0,1,1,1),
						),
						array(
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,1,1,1,1,0,0,0,0),
							array(1,1,0,0,1,1,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
						),
					),
		'S'	=>		array(
						array(
							array(0,0,1,1,1,1,1,0,0),
							array(0,1,0,0,0,0,0,1,0),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,0,1,1,1,1,1,0,0),
							array(0,0,0,0,0,0,0,1,0),
							array(0,0,0,0,0,0,0,0,1),
							array(0,0,0,0,0,0,0,0,1),
							array(0,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,1,0),
							array(0,0,1,1,1,1,1,0,0),
						),
						array(
							array(0,0,1,1,1,1,1,0,1),
							array(0,1,0,0,0,0,0,1,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(0,0,1,1,1,1,1,0,0),
							array(0,0,0,0,0,0,0,1,0),
							array(0,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,1,0,0,0,0,0,1,0),
							array(1,0,1,1,1,1,1,0,0),
						),
						array(
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,1,1,1,1,0,0,0,0),
							array(1,0,0,0,0,1,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,1,0,0,0,0,0,0,0),
							array(0,1,1,1,1,0,0,0,0),
							array(0,0,0,0,0,1,0,0,0),
							array(1,0,0,0,1,1,0,0,0),
							array(0,1,1,1,1,0,0,0,0),
						),
					),
		'T'	=>		array(
						array(
							array(1,1,1,1,1,1,1,1,1),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
						),
						array(
							array(1,1,1,1,1,1,1,1,1),
							array(1,0,0,0,1,0,0,0,1),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,1,1,1,0,0,0),
						),
						array(
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,1,1,1,1,1,1,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,1,0,0,0),
							array(0,0,0,0,0,1,1,1,0),
						),
					),
		'U'	=>		array(
						array(
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,1,0),
							array(0,0,1,1,1,1,1,0,0),
						),
						array(
							array(1,0,0,0,0,0,0,0,0),
							array(1,1,1,0,0,0,1,1,1),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,1,0,0,0,1,1,0),
							array(0,0,1,1,1,1,1,0,0),
						),
						array(
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,1,0,0,0,0,0,1),
							array(0,0,1,0,0,0,0,0,1),
							array(0,0,1,0,0,0,0,0,1),
							array(0,0,1,0,0,0,0,0,1),
							array(0,0,1,0,0,0,0,0,1),
							array(0,0,1,0,0,0,0,1,1),
							array(0,0,1,1,0,0,1,1,1),
							array(0,0,0,1,1,1,1,0,1),
						),
					),
		'V'	=>		array(
						array(
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,0,1,0,0,0,1,0,0),
							array(0,0,1,0,0,0,1,0,0),
							array(0,0,1,0,0,0,1,0,0),
							array(0,0,1,0,0,0,1,0,0),
							array(0,0,0,1,0,1,0,0,0),
							array(0,0,0,1,0,1,0,0,0),
							array(0,0,0,1,0,1,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
						),
						array(
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(1,1,1,0,0,0,1,1,1),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,0,1,0,0,0,1,0,0),
							array(0,0,1,0,0,0,1,0,0),
							array(0,0,1,0,0,0,1,0,0),
							array(0,0,1,0,0,0,1,0,0),
							array(0,0,0,1,0,1,0,0,0),
							array(0,0,0,1,0,1,0,0,0),
							array(0,0,0,1,0,1,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
						),
						array(
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,1,0,0,0,1,0,0),
							array(0,0,1,0,0,0,1,0,0),
							array(0,0,1,0,0,0,1,0,0),
							array(0,0,1,0,0,0,1,0,0),
							array(0,0,0,1,0,1,0,0,0),
							array(0,0,0,1,0,1,0,0,0),
							array(0,0,0,1,0,1,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
						),
					),
		'W'	=>		array(
						array(
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,1,0,0,0,1),
							array(1,0,0,0,1,0,0,0,1),
							array(1,0,0,1,0,1,0,0,1),
							array(1,0,0,1,0,1,0,0,1),
							array(1,0,0,1,0,1,0,0,1),
							array(1,0,1,0,0,0,1,0,1),
							array(1,0,1,0,0,0,1,0,1),
							array(1,0,1,0,0,0,1,0,1),
							array(1,1,0,0,0,0,0,1,1),
							array(1,1,0,0,0,0,0,1,1),
						),
						array(
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(1,1,1,0,0,0,1,1,1),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,1,0,0,1,0),
							array(0,1,0,0,1,0,0,1,0),
							array(0,1,0,1,1,1,0,1,0),
							array(0,1,0,1,0,1,0,1,0),
							array(0,1,1,1,0,1,1,1,0),
							array(0,1,1,0,0,0,1,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,0,0,0,0,0,0,0,0),
						),
						array(
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,1,0,0,1,0),
							array(0,1,0,0,1,0,0,1,0),
							array(0,1,0,1,1,1,0,1,0),
							array(0,1,0,1,0,1,0,1,0),
							array(0,1,1,1,0,1,1,1,0),
							array(0,1,1,0,0,0,1,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,0,0,0,0,0,0,0,0),
						),
					),
		'X'	=>		array(
						array(
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,0,1,0,0,0,1,0,0),
							array(0,0,0,1,0,1,0,0,0),
							array(0,0,0,1,0,1,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,1,0,1,0,0,0),
							array(0,0,0,1,0,1,0,0,0),
							array(0,0,1,0,0,0,1,0,0),
							array(0,1,0,0,0,0,1,0,0),
							array(0,1,0,0,0,0,0,1,0),
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
						),
						array(
							array(0,0,0,0,0,0,0,0,0),
							array(1,1,1,0,0,0,1,1,1),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,0,1,0,0,0,1,0,0),
							array(0,0,0,1,0,1,0,0,0),
							array(0,0,0,1,0,1,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,1,0,1,0,0,0),
							array(0,0,0,1,0,1,0,0,0),
							array(0,0,1,0,0,0,1,0,0),
							array(0,1,0,0,0,0,1,0,0),
							array(0,1,0,0,0,0,0,1,0),
							array(1,1,1,0,0,0,1,1,1),
							array(0,0,0,0,0,0,0,0,0),
						),
						array(
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,1,0,0,0,1,1,0),
							array(0,0,1,1,0,1,1,0,0),
							array(0,0,0,1,1,1,0,0,0),
							array(0,0,0,1,1,1,0,0,0),
							array(0,0,1,1,0,1,1,0,0),
							array(0,1,1,0,0,0,1,1,0),
							array(0,0,0,0,0,0,0,0,0),
						),
					),
		'Y'	=>		array(
						array(
							array(1,0,0,0,0,0,0,0,1),
							array(1,0,0,0,0,0,0,0,1),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,0,1,0,0,0,1,0,0),
							array(0,0,1,0,0,0,1,0,0),
							array(0,0,0,1,0,1,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
						),
						array(
							array(0,0,0,0,0,0,0,0,0),
							array(1,1,1,0,0,0,1,1,1),
							array(0,1,0,0,0,0,0,1,0),
							array(0,1,0,0,0,0,0,1,0),
							array(0,0,1,0,0,0,1,0,0),
							array(0,0,1,0,0,0,1,0,0),
							array(0,0,0,1,0,1,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,1,1,1,0,0,0),
						),
						array(
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,1,0,0,0,0,1),
							array(0,0,0,1,1,0,0,0,1),
							array(0,0,0,0,1,0,0,1,1),
							array(0,0,0,0,1,1,0,1,0),
							array(0,0,0,0,0,1,1,1,0),
							array(0,0,0,0,0,0,1,0,0),
							array(0,0,0,0,0,1,1,0,0),
							array(0,0,0,0,0,1,0,0,0),
							array(0,0,0,0,1,1,0,0,0),
							array(0,0,1,1,1,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
						),
					),
		'Z'	=>		array(
						array(
							array(1,1,1,1,1,1,1,1,1),
							array(1,0,0,0,0,0,0,0,1),
							array(0,0,0,0,0,0,0,0,1),
							array(0,0,0,0,0,0,0,1,0),
							array(0,0,0,0,0,0,1,0,0),
							array(0,0,0,0,0,1,0,0,0),
							array(0,0,0,0,0,1,0,0,0),
							array(0,0,0,0,1,0,0,0,0),
							array(0,0,0,1,0,0,0,0,0),
							array(0,0,0,1,0,0,0,0,0),
							array(0,0,1,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,1),
							array(1,1,1,1,1,1,1,1,1),
						),
						array(
							array(1,1,1,1,1,1,1,1,1),
							array(0,0,0,0,0,0,0,0,1),
							array(0,0,0,0,0,0,0,0,1),
							array(0,0,0,0,0,0,0,1,0),
							array(0,0,0,0,0,0,1,0,0),
							array(0,0,0,0,0,1,0,0,0),
							array(0,0,0,0,0,1,0,0,0),
							array(0,0,1,1,1,1,1,0,0),
							array(0,0,0,1,0,0,0,0,0),
							array(0,0,0,1,0,0,0,0,0),
							array(0,0,1,0,0,0,0,0,0),
							array(0,1,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,0,0,0,0,0,0,0,0),
							array(1,1,1,1,1,1,1,1,1),
						),
						array(
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,0,0,0,0,0,0,0,0),
							array(0,1,1,1,1,1,1,1,0),
							array(0,0,0,0,0,1,1,0,0),
							array(0,0,0,0,1,1,0,0,0),
							array(0,0,0,1,1,0,0,0,0),
							array(0,0,1,1,0,0,0,0,0),
							array(0,0,1,0,0,0,0,0,0),
							array(0,1,1,1,1,1,1,1,0),
						),
					),
				);
		return array(
			'width'		=> 9,
			'height'	=> 15,
			'data'		=> array(

			'A' =>	$chars['A'][mt_rand(0, min(sizeof($chars['A']), $config['captcha_gd_fonts']) -1)],
			'B' =>	$chars['B'][mt_rand(0, min(sizeof($chars['B']), $config['captcha_gd_fonts']) -1)],
			'C' =>	$chars['C'][mt_rand(0, min(sizeof($chars['C']), $config['captcha_gd_fonts']) -1)],
			'D' =>	$chars['D'][mt_rand(0, min(sizeof($chars['D']), $config['captcha_gd_fonts']) -1)],
			'E' =>	$chars['E'][mt_rand(0, min(sizeof($chars['E']), $config['captcha_gd_fonts']) -1)],
			'F' =>	$chars['F'][mt_rand(0, min(sizeof($chars['F']), $config['captcha_gd_fonts']) -1)],
			'G' =>	$chars['G'][mt_rand(0, min(sizeof($chars['G']), $config['captcha_gd_fonts']) -1)],
			'H' =>	$chars['H'][mt_rand(0, min(sizeof($chars['H']), $config['captcha_gd_fonts']) -1)],
			'I' =>	$chars['I'][mt_rand(0, min(sizeof($chars['I']), $config['captcha_gd_fonts']) -1)],
			'J' =>	$chars['J'][mt_rand(0, min(sizeof($chars['J']), $config['captcha_gd_fonts']) -1)],
			'K' =>	$chars['K'][mt_rand(0, min(sizeof($chars['K']), $config['captcha_gd_fonts']) -1)],
			'L' =>	$chars['L'][mt_rand(0, min(sizeof($chars['L']), $config['captcha_gd_fonts']) -1)],
			'M' =>	$chars['M'][mt_rand(0, min(sizeof($chars['M']), $config['captcha_gd_fonts']) -1)],
			'N' =>	$chars['N'][mt_rand(0, min(sizeof($chars['N']), $config['captcha_gd_fonts']) -1)],
			'O' =>	$chars['O'][mt_rand(0, min(sizeof($chars['O']), $config['captcha_gd_fonts']) -1)],
			'P' =>	$chars['P'][mt_rand(0, min(sizeof($chars['P']), $config['captcha_gd_fonts']) -1)],
			'Q' =>	$chars['Q'][mt_rand(0, min(sizeof($chars['Q']), $config['captcha_gd_fonts']) -1)],
			'R' =>	$chars['R'][mt_rand(0, min(sizeof($chars['R']), $config['captcha_gd_fonts']) -1)],
			'S' =>	$chars['S'][mt_rand(0, min(sizeof($chars['S']), $config['captcha_gd_fonts']) -1)],
			'T' =>	$chars['T'][mt_rand(0, min(sizeof($chars['T']), $config['captcha_gd_fonts']) -1)],
			'U' =>	$chars['U'][mt_rand(0, min(sizeof($chars['U']), $config['captcha_gd_fonts']) -1)],
			'V' =>	$chars['V'][mt_rand(0, min(sizeof($chars['V']), $config['captcha_gd_fonts']) -1)],
			'W' =>	$chars['W'][mt_rand(0, min(sizeof($chars['W']), $config['captcha_gd_fonts']) -1)],
			'X' =>	$chars['X'][mt_rand(0, min(sizeof($chars['X']), $config['captcha_gd_fonts']) -1)],
			'Y' =>	$chars['Y'][mt_rand(0, min(sizeof($chars['Y']), $config['captcha_gd_fonts']) -1)],
			'Z' =>	$chars['Z'][mt_rand(0, min(sizeof($chars['Z']), $config['captcha_gd_fonts']) -1)],

			'1' => array(
				array(0,0,0,1,1,0,0,0,0),
				array(0,0,1,0,1,0,0,0,0),
				array(0,1,0,0,1,0,0,0,0),
				array(0,0,0,0,1,0,0,0,0),
				array(0,0,0,0,1,0,0,0,0),
				array(0,0,0,0,1,0,0,0,0),
				array(0,0,0,0,1,0,0,0,0),
				array(0,0,0,0,1,0,0,0,0),
				array(0,0,0,0,1,0,0,0,0),
				array(0,0,0,0,1,0,0,0,0),
				array(0,0,0,0,1,0,0,0,0),
				array(0,0,0,0,1,0,0,0,0),
				array(0,0,0,0,1,0,0,0,0),
				array(0,0,0,0,1,0,0,0,0),
				array(0,1,1,1,1,1,1,1,0),
			),
			'2' => array(    // New '2' supplied by Anon
				array(0,0,0,1,1,1,0,0,0),
				array(0,0,1,0,0,0,1,0,0),
				array(0,1,0,0,0,0,1,1,0),
				array(0,0,0,0,0,0,0,0,1),
				array(0,0,0,0,0,0,0,0,1),
				array(0,0,0,0,0,0,0,1,1),
				array(0,0,0,0,0,0,0,1,0),
				array(0,0,0,0,0,0,1,0,0),
				array(0,0,0,0,0,1,0,0,0),
				array(0,0,0,0,1,0,0,0,0),
				array(0,0,0,1,0,0,0,0,0),
				array(0,0,1,0,0,0,0,0,0),
				array(0,1,0,0,0,0,0,0,0),
				array(1,1,1,1,1,1,1,1,1),
				array(0,0,0,0,0,0,0,0,0),
			),
			'3' => array(
				array(0,0,1,1,1,1,1,0,0),
				array(0,1,0,0,0,0,0,1,0),
				array(1,0,0,0,0,0,0,0,1),
				array(0,0,0,0,0,0,0,0,1),
				array(0,0,0,0,0,0,0,0,1),
				array(0,0,0,0,0,0,0,0,1),
				array(0,0,0,0,0,0,0,1,0),
				array(0,0,0,0,0,1,1,0,0),
				array(0,0,0,0,0,0,0,1,0),
				array(0,0,0,0,0,0,0,0,1),
				array(0,0,0,0,0,0,0,0,1),
				array(0,0,0,0,0,0,0,0,1),
				array(1,0,0,0,0,0,0,0,1),
				array(0,1,0,0,0,0,0,1,0),
				array(0,0,1,1,1,1,1,0,0),
			),
			'4' => array(
				array(0,0,0,0,0,0,1,1,0),
				array(0,0,0,0,0,1,0,1,0),
				array(0,0,0,0,1,0,0,1,0),
				array(0,0,0,1,0,0,0,1,0),
				array(0,0,1,0,0,0,0,1,0),
				array(0,1,0,0,0,0,0,1,0),
				array(1,0,0,0,0,0,0,1,0),
				array(1,0,0,0,0,0,0,1,0),
				array(1,1,1,1,1,1,1,1,1),
				array(0,0,0,0,0,0,0,1,0),
				array(0,0,0,0,0,0,0,1,0),
				array(0,0,0,0,0,0,0,1,0),
				array(0,0,0,0,0,0,0,1,0),
				array(0,0,0,0,0,0,0,1,0),
				array(0,0,0,0,0,0,0,1,0),
			),
			'5' => array(
				array(1,1,1,1,1,1,1,1,1),
				array(1,0,0,0,0,0,0,0,0),
				array(1,0,0,0,0,0,0,0,0),
				array(1,0,0,0,0,0,0,0,0),
				array(1,0,0,0,0,0,0,0,0),
				array(0,1,0,0,0,0,0,0,0),
				array(0,0,1,1,1,1,1,0,0),
				array(0,0,0,0,0,0,0,1,0),
				array(0,0,0,0,0,0,0,0,1),
				array(0,0,0,0,0,0,0,0,1),
				array(0,0,0,0,0,0,0,0,1),
				array(0,0,0,0,0,0,0,0,1),
				array(1,0,0,0,0,0,0,0,1),
				array(0,1,0,0,0,0,0,1,0),
				array(0,0,1,1,1,1,1,0,0),
			),
			'6' => array(
				array(0,0,1,1,1,1,1,0,0),
				array(0,1,0,0,0,0,0,1,0),
				array(1,0,0,0,0,0,0,0,1),
				array(1,0,0,0,0,0,0,0,1),
				array(1,0,0,0,0,0,0,0,0),
				array(1,0,0,0,0,0,0,0,0),
				array(1,0,0,1,1,1,1,0,0),
				array(1,0,1,0,0,0,0,1,0),
				array(1,1,0,0,0,0,0,0,1),
				array(1,0,0,0,0,0,0,0,1),
				array(1,0,0,0,0,0,0,0,1),
				array(1,0,0,0,0,0,0,0,1),
				array(1,0,0,0,0,0,0,0,1),
				array(0,1,0,0,0,0,0,1,0),
				array(0,0,1,1,1,1,1,0,0),
			),
			'7' => array(
				array(1,1,1,1,1,1,1,1,1),
				array(0,0,0,0,0,0,0,0,1),
				array(0,0,0,0,0,0,0,1,0),
				array(0,0,0,0,0,0,0,1,0),
				array(0,0,0,0,0,0,1,0,0),
				array(0,0,0,0,0,1,0,0,0),
				array(0,0,0,0,0,1,0,0,0),
				array(0,0,0,0,1,0,0,0,0),
				array(0,0,0,1,0,0,0,0,0),
				array(0,0,0,1,0,0,0,0,0),
				array(0,0,1,0,0,0,0,0,0),
				array(0,1,0,0,0,0,0,0,0),
				array(0,1,0,0,0,0,0,0,0),
				array(1,0,0,0,0,0,0,0,0),
				array(1,0,0,0,0,0,0,0,0),
			),
			'8' => array(
				array(0,0,1,1,1,1,1,0,0),
				array(0,1,0,0,0,0,0,1,0),
				array(1,0,0,0,0,0,0,0,1),
				array(1,0,0,0,0,0,0,0,1),
				array(1,0,0,0,0,0,0,0,1),
				array(1,0,0,0,0,0,0,0,1),
				array(0,1,0,0,0,0,0,1,0),
				array(0,0,1,1,1,1,1,0,0),
				array(0,1,0,0,0,0,0,1,0),
				array(1,0,0,0,0,0,0,0,1),
				array(1,0,0,0,0,0,0,0,1),
				array(1,0,0,0,0,0,0,0,1),
				array(1,0,0,0,0,0,0,0,1),
				array(0,1,0,0,0,0,0,1,0),
				array(0,0,1,1,1,1,1,0,0),
			),
			'9' => array(
				array(0,0,1,1,1,1,1,0,0),
				array(0,1,0,0,0,0,0,1,0),
				array(1,0,0,0,0,0,0,0,1),
				array(1,0,0,0,0,0,0,0,1),
				array(1,0,0,0,0,0,0,0,1),
				array(1,0,0,0,0,0,0,0,1),
				array(1,0,0,0,0,0,0,1,1),
				array(0,1,0,0,0,0,1,0,1),
				array(0,0,1,1,1,1,0,0,1),
				array(0,0,0,0,0,0,0,0,1),
				array(0,0,0,0,0,0,0,0,1),
				array(1,0,0,0,0,0,0,0,1),
				array(1,0,0,0,0,0,0,0,1),
				array(0,1,0,0,0,0,0,1,0),
				array(0,0,1,1,1,1,1,0,0),
			),
			)
		);
	}
}
